﻿using System;
using System.Collections.Generic;
using System.Globalization;
using System.Text;
using System.Windows.Forms;
using System.IO;

using KONICAMINOLTA.OpenAPISDK.SequenceSetupLibrary;
using KONICAMINOLTA.OpenAPISDK.SequenceSetupLibrary.IF.DeviceSettingNS;
using KONICAMINOLTA.OpenAPISDK.SequenceSetupLibrary.IF.ApplicationSettingNS;
using KONICAMINOLTA.OpenAPISDK.SequenceSetupLibrary.IF.ApplicationInfoNS;
using KONICAMINOLTA.OpenAPISDK.CommonLibrary.Facade;
using KONICAMINOLTA.OpenAPISDK.SequenceAuthLibrary.Facade.AppRequests;

namespace OpenAPI_App_Manager
{
    public class OpenAPIWrapper
    {
        //All OpenAPI activity (per MFP) is performed within this class.  This avoids conflicts for access to the MFP.
        //It is essentially just a wrapper around a single instance of the Setup MainControlManager

        private MFPConfig mfp;

        // Added for Bi Directional Authentication.
        // Certificate Path
        private const string CERTIFICATEFILEPATH =  @"Data\Pem\syoumei.p12";

        public MFPConfig MFP
        {
            get
            {
                return mfp;
            }
            set
            {   
                mfp = MFP;
                CreateSetupManager();
            }
        }

        public MainControlManagerFacade SetupMainControlManager;

        private object activityLock = new object();

        public OpenAPIWrapper(MFPConfig mfp)
        {
            this.mfp = mfp;
            CreateSetupManager();
        }

        private static T StringToEnum<T>(string EnumName)
        {
            return (T)Enum.Parse(typeof(T), EnumName);
        }

        private void CreateSetupManager()
        {
            //This method creates a standard "Setup MainControlManager", used for almost all activities in this application.
            SetupMainControlManager = new MainControlManagerFacade(mfp.Address);

            // Added For BiDirectional Authentication.
            string certifaicetFilePath = Application.StartupPath + Path.DirectorySeparatorChar + CERTIFICATEFILEPATH;

            //InitCommonLibrarySetting is the method that initializes Common library setting. 
            //InitCommonLibrarySetting(
            // loglevel: This creates an "OpenAPISDKx.log" file for communication that occurs. LogLevel determines the level of detail in the log files.  See the OpenAPI documentation for further details.
            // outputFoldel: Output folder of "OpenAPISDKx.log" file
            //)
            SetupMainControlManager.InitCommonLibrarySetting(
                LogLevel.OpenAPIMessageOnly,
                AppDomain.CurrentDomain.BaseDirectory);

            //SetupFunctionVersionType defines the version of the setup application.
            SetupFunctionVersionType functionVersionType;
            try
            {
                string functionVersion = "v"+ mfp.FunctionVersion.Setup.Highest.ToString("0.0");
                string szFunctionVersion = functionVersion.Replace(".", "_");
                functionVersionType = StringToEnum<SetupFunctionVersionType>(szFunctionVersion);
            }
            catch
            {
                //The attempt to convert the string to an enum failed, perhaps because the enum doesn't support the version in question.
                //We'll hardcode a reasonable value and hope it's okay.
                //functionVersionType = SetupFunctionVersionType.v2_0;
                functionVersionType = SetupFunctionVersionType.v4_0;
            }

            //InitDeviceCommunicationSetting is the method that initializes the setting of OpenAPI communication with an MFP. Configures basic settings required for a communication with an MFP. 
            //InitDeviceCommunicationSetting(
            // functionVersionType: Setup FunctionVersion Type
            // openAPIPort: The port number where MFP listens OpenAPI Message
            // openAPITimeout: Timeout is measured in milliseconds, so the multiplier of 1000 here allows the user to select seconds.
            // useSSL: A value indicating whether SSL is being used
            // openAPIUser: OpenAPI User Name (When OpenAPI Authentication is enabled in MFP, this parameter is necessary.)
            // openAPIPassword: OpenAPI User Password (When OpenAPI Authentication is enabled in MFP, this parameter is necessary.)
            // (Note that the above OpenAPI Authentication setting parameter (openAPIUser and openAPIPassword) is NOT related to the Authentication function of the MFP.  It is an optional username or password in order
            //  to establish OpenAPI communication with this MFP.  It is configured from the OpenAPI settings on the MFP control panel in Administrator mode.
            //  If OpenAPI Authentication is not enabled, this setting can be omitted, however it does not cause problems to pass any values, and so
            //  there is no need to make this setting conditional.)
            // deviceLoginUser: Login User Name (It is necessary to login by the Administrator for using the setup message. Sets "Admin" to this parameter.)
            // deviceLoginPassword: Login User Passwrod (It is necessary to login by the Administrator for using the setup message. Sets Administrator password to this parameter.)
            //)
            //Note that these settings are for the setup application to communicate with the MFP and do not nessarily imply that the MFP will communicate
            //with these settings to any listening threads (there are no listening threads in this application - please see some sample applications except setup applications)
            SetupMainControlManager.InitDeviceCommunicationSetting(
                functionVersionType, 
                certifaicetFilePath, // Added for BiDirectional Authentication.
                mfp.Port,
                60000, //60 seconds
                mfp.SSL,
                mfp.OpenAPIUsername,
                mfp.OpenAPIPassword,
                "Admin",
                mfp.Password);
        }

        internal ConnectionResult GetApplicationList(out ApplicationInfoCollection AppInfos)
        {
            lock (activityLock)
            {
                try
                {
                    return SetupMainControlManager.GetApplicationList(out AppInfos);
                }
                catch
                {
                    AppInfos = new ApplicationInfoCollection();
                    return ConnectionResult.Nack;
                }
            }
        }

        internal ConnectionResult Login()
        {
            lock (activityLock)
                return SetupMainControlManager.Login();
        }

        internal ConnectionResult Logout()
        {
            lock (activityLock)
                return SetupMainControlManager.Logout();
        }

        internal ConnectionResult DeleteApplication(int AppNo)
        {
            lock (activityLock)
                return SetupMainControlManager.DeleteApplication(AppNo);
        }

        internal ConnectionResult GetScanApplication(int AppNo, string AppID, out ScanTypeDApplicationSetting AppSetting)
        {
            lock (activityLock)
                return SetupMainControlManager.GetScanApplication(AppNo, AppID, out AppSetting);
        }

        internal ConnectionResult GetScanApplication(int AppNo, string AppID, out ScanTypeSApplicationSetting AppSetting)
        {
            lock (activityLock)
                return SetupMainControlManager.GetScanApplication(AppNo, AppID, out AppSetting);
        }

        internal ConnectionResult GetScanApplication(int AppNo, string AppID, out ScanBrowserApplicationSetting AppSetting)
        {
            lock (activityLock)
                return SetupMainControlManager.GetScanApplication(AppNo, AppID, out AppSetting);
        }

        internal ConnectionResult GetPullApplication(int AppNo, string AppID, out PullApplicationSetting AppSetting)
        {
            lock (activityLock)
                return SetupMainControlManager.GetPullApplication(AppNo, AppID, out AppSetting);
        }

        internal ConnectionResult GetPullBrowserApplication(int AppNo, string AppID, out PullBrowserApplicationSetting AppSetting)
        {
            lock (activityLock)
                return SetupMainControlManager.GetPullBrowserApplication(AppNo, AppID, out AppSetting);
        }

        internal ConnectionResult GetAuthApplication(int AppNo, string AppID, out AuthApplicationSetting AppSetting)
        {
            lock (activityLock)
                return SetupMainControlManager.GetAuthApplication(AppNo, AppID, out AppSetting);
        }

        internal ConnectionResult GetAuthBrowserApplication(int AppNo, string AppID, out AuthBrowserApplicationSetting AppSetting)
        {
            lock (activityLock)
                return SetupMainControlManager.GetAuthBrowserApplication(AppNo, AppID, out AppSetting);
        }

        internal ConnectionResult GetBrowserLauncherApplication(int AppNo, string AppID, out BrowserLauncherApplicationSetting AppSetting)
        {
            lock (activityLock)
                return SetupMainControlManager.GetBrowserLauncherApplication(AppNo, AppID, out AppSetting);
        }

        internal ConnectionResult GetOtherAApplication(int AppNo, string AppID, out OtherAApplicationSetting AppSetting)
        {
            lock (activityLock)
                return SetupMainControlManager.GetOtherAApplication(AppNo, AppID, out AppSetting);
        }

        internal ConnectionResult GetScreenSaverApplication(int AppNo, string AppID, out ScreenSaverApplicationSetting AppSetting)
        {
            lock (activityLock)
                return SetupMainControlManager.GetScreenSaverApplication(AppNo, AppID, out AppSetting);
        }

        internal ConnectionResult GetGeneralBrowserApplication(int AppNo, string AppID, out GeneralBrowserApplicationSetting AppSetting)
        {
            lock (activityLock)
                return SetupMainControlManager.GetGeneralBrowserApplication(AppNo, AppID, out AppSetting);
        }

        internal ConnectionResult GetDeviceInfo()
        {
            lock (activityLock)
                try
                {
                    return SetupMainControlManager.GetDeviceInfo();
                }
                catch {
                    return ConnectionResult.Nack;
                 }
        }

        internal ConnectionResult SetPullBrowserApplication(PullBrowserApplicationSetting pullBrowserApplicationSetting, out int registerNo)
        {
            lock (activityLock)
                return SetupMainControlManager.SetPullBrowserApplication(pullBrowserApplicationSetting, out registerNo);
        }

        internal int FindAppNoByName(string AppName)
        {
            lock (activityLock)
            {
                try
                {
                    KONICAMINOLTA.OpenAPISDK.SequenceSetupLibrary.IF.ApplicationInfoNS.ApplicationInfoCollection appList;
                    SetupMainControlManager.GetApplicationList(out appList);
                    foreach (KONICAMINOLTA.OpenAPISDK.SequenceSetupLibrary.IF.ApplicationInfoNS.ApplicationInfo app in appList)
                    {
                        if (app.Name == AppName)
                            return app.No;
                    }
                    return -1;
                }
                catch
                {
                    return -1;
                }
            }
        }

        internal ConnectionResult SetExternalAppsButtonName(string PanelButtonText)
        {
            lock (activityLock)
                return SetupMainControlManager.SetExternalAppsButtonName(PanelButtonText);
        }

        internal ConnectionResult GetExternalAppsButtonName(out string PanelButtonText)
        {
            lock (activityLock)
                return SetupMainControlManager.GetExternalAppsButtonName(out PanelButtonText);
        }

        internal ConnectionResult SetAuthenticationMode(AuthSettings Settings, AuthAppDetails AuthApp)
        {
            lock (activityLock)
            {
                try
                {
                    //This method sets the Authentication Mode of the MFP.  Note that this is done using the Authentication sub-API, not the Setup API
                    //Firstly, we create the "ChangeAuthenticationSetting" request.
                    ChangeAuthenticationSettingRequest authChange = new ChangeAuthenticationSettingRequest();

                    //This request, because it is not using a pre-established Setup Facade, needs some basic knowledge about the MFP
                    authChange.ClientProtocol.OpenAPIAuthentication.LoginName = MFP.OpenAPIUsername;
                    authChange.ClientProtocol.OpenAPIAuthentication.Password = MFP.OpenAPIPassword;
                    if (AuthApp.BrowserAuth)
                        authChange.ClientProtocol.FunctionVersion.ApplicationType = KONICAMINOLTA.OpenAPISDK.CommonLibrary.Facade.ApplicationType.AuthenticationBrowser;
                    else
                        authChange.ClientProtocol.FunctionVersion.ApplicationType = KONICAMINOLTA.OpenAPISDK.CommonLibrary.Facade.ApplicationType.Authentication;
                    authChange.ClientProtocol.FunctionVersion.Major = AuthApp.FunctionVersionMajor;
                    authChange.ClientProtocol.FunctionVersion.Minor = AuthApp.FunctionVersionMinor;
                    authChange.ClientProtocol.Port = MFP.Port;
                    authChange.ClientProtocol.Ssl = MFP.SSL;
                    authChange.ClientProtocol.RequestTimeout = 40000; //Milliseconds

                    //Due to an SDK bug in SDK3.7, the above RequestTimeout value is ignored.  It can also be set by setting the SDKServicePointManager directly.
                    SDKServicePointManager.Client.RequestTimeout = 40000;

                    //Now, we define what exactly we're requesting
                    authChange.RelayAuthentication = Settings.Enable; //Defines whether "Relay" (OpenAPI) Authentication is on or off
                    if (authChange.RelayAuthentication) //If Authentication is on
                    {
                        authChange.AllowBoxAdmin = Settings.BoxAdmin;                   //Defines whether the special "Box Administrator" account is allowed
                        if (Settings.PublicMode == AuthSettings.PublicAuthMode.Off)
                            authChange.AllowPublicUser = false;
                        else
                        {
                            authChange.AllowPublicUser = true;                                                                        //Defines whether the special "Public User" (access without communication to application) is allowed.
                            authChange.LoginPublicUserAfterLogout = (Settings.PublicMode == AuthSettings.PublicAuthMode.OnByDefault); //Defines whether the Public User is automatically logged in when users log out
                        }
                        authChange.AllowChangePublicUserSetting = Settings.AdminMayEdit;
                    }

                    bool bSuccess = authChange.Send(MFP.Address, MFP.Password); //Submit the request
                    if (!bSuccess)
                    {
                        return ConnectionResult.Nack;
                    }
                }
                catch
                {
                    return ConnectionResult.Nack;
                }
            }
            return ConnectionResult.Ack;
        }

        internal ConnectionResult GetAuthenticationMode(out AuthSettings authSet)
        {
            lock (activityLock)
            {
                //Note: If the current Authentication mode is something other than OpenAPI Authentication (e.g. "External Server"
                //or "Internal Authentication"), then it is treated as "off" by this method.
                authSet = null;

                KONICAMINOLTA.OpenAPISDK.SequenceSetupLibrary.IF.ResGetApplicationListNS.ResGetApplicationList resAppList = new KONICAMINOLTA.OpenAPISDK.SequenceSetupLibrary.IF.ResGetApplicationListNS.ResGetApplicationList();
                try
                {
                    KONICAMINOLTA.OpenAPISDK.SequenceSetupLibrary.ConnectionResult cr = SetupMainControlManager.GetDeviceInfo();
                    if (cr == KONICAMINOLTA.OpenAPISDK.SequenceSetupLibrary.ConnectionResult.Ack)
                    {
                        authSet = new AuthSettings(
                            (SetupMainControlManager.DeviceInfo.AuthenticationInformation.AuthSetting == "4"),
                            (SetupMainControlManager.DeviceInfo.AuthenticationInformation.BoxAdmin == "1"),
                             AuthSettings.PublicAuthMode.Off,
                             new FunctionLimitationSetting(),
                            (SetupMainControlManager.DeviceInfo.AuthenticationInformation.PublicInfo == "1"));

                        if (SetupMainControlManager.DeviceInfo.AuthenticationInformation.PublicSetting == "1")
                            authSet.PublicMode = AuthSettings.PublicAuthMode.On;
                        else if (SetupMainControlManager.DeviceInfo.AuthenticationInformation.PublicSetting == "3")
                            authSet.PublicMode = AuthSettings.PublicAuthMode.OnByDefault;

                        cr = GetFunctionLimitationSetting(out authSet.PublicUserLimits);
                        if (cr == ConnectionResult.Nack)
                            return ConnectionResult.Partial;
                    }
                    else
                        return cr;
                }
                catch
                {
                    return ConnectionResult.Nack;
                }
                return ConnectionResult.Ack;
            }
        }

        internal ConnectionResult GetFunctionLimitationSetting(out FunctionLimitationSetting functionLimitationSetting)
        {
            lock (activityLock)
            {
                return SetupMainControlManager.GetFunctionLimitationSetting(out functionLimitationSetting);
            }
        }

        internal ConnectionResult IsDeviceAlive()
        {
            lock (activityLock)
            {
                try
                {
                    return SetupMainControlManager.IsDeviceAlive();
                }
                catch
                {
                    return ConnectionResult.Nack;
                }
            }
        }
    }
}
